<?php

/**
 * CampusEats Helper Functions
 */

use Illuminate\Support\Facades\Cache;

if (!function_exists('format_naira')) {
    /**
     * Format amount as Naira currency.
     */
    function format_naira(float|int $amount, bool $showSymbol = true): string
    {
        $formatted = number_format($amount, 2);
        return $showSymbol ? "₦{$formatted}" : $formatted;
    }
}

if (!function_exists('format_phone')) {
    /**
     * Format phone number for display.
     */
    function format_phone(string $phone): string
    {
        // Remove non-numeric characters
        $phone = preg_replace('/[^0-9]/', '', $phone);
        
        // Format as Nigerian number
        if (strlen($phone) === 13 && str_starts_with($phone, '234')) {
            return '+234 ' . substr($phone, 3, 3) . ' ' . substr($phone, 6, 3) . ' ' . substr($phone, 9);
        }
        
        if (strlen($phone) === 11 && str_starts_with($phone, '0')) {
            return substr($phone, 0, 4) . ' ' . substr($phone, 4, 3) . ' ' . substr($phone, 7);
        }
        
        return $phone;
    }
}

if (!function_exists('normalize_phone')) {
    /**
     * Normalize phone number to +234 format.
     */
    function normalize_phone(string $phone): string
    {
        $phone = preg_replace('/[^0-9]/', '', $phone);
        
        if (str_starts_with($phone, '0')) {
            return '+234' . substr($phone, 1);
        }
        
        if (!str_starts_with($phone, '234')) {
            return '+234' . $phone;
        }
        
        return '+' . $phone;
    }
}

if (!function_exists('generate_order_number')) {
    /**
     * Generate unique order number.
     */
    function generate_order_number(): string
    {
        $prefix = 'CE';
        $date = now()->format('ymd');
        $random = strtoupper(substr(md5(uniqid(mt_rand(), true)), 0, 6));
        
        return "{$prefix}{$date}{$random}";
    }
}

if (!function_exists('calculate_distance')) {
    /**
     * Calculate distance between two coordinates using Haversine formula.
     * Returns distance in kilometers.
     */
    function calculate_distance(
        float $lat1,
        float $lng1,
        float $lat2,
        float $lng2
    ): float {
        $earthRadius = 6371; // Radius of Earth in kilometers
        
        $latDiff = deg2rad($lat2 - $lat1);
        $lngDiff = deg2rad($lng2 - $lng1);
        
        $a = sin($latDiff / 2) * sin($latDiff / 2) +
             cos(deg2rad($lat1)) * cos(deg2rad($lat2)) *
             sin($lngDiff / 2) * sin($lngDiff / 2);
        
        $c = 2 * atan2(sqrt($a), sqrt(1 - $a));
        
        return round($earthRadius * $c, 2);
    }
}

if (!function_exists('is_peak_hour')) {
    /**
     * Check if current time is within peak hours.
     */
    function is_peak_hour(): bool
    {
        $now = now();
        $hour = (int) $now->format('H');
        $minute = (int) $now->format('i');
        $currentTime = $hour * 60 + $minute;
        
        // Peak hours: 12:00-14:00 and 18:00-21:00
        $peakPeriods = [
            ['start' => 12 * 60, 'end' => 14 * 60],      // 12:00 - 14:00
            ['start' => 18 * 60, 'end' => 21 * 60],      // 18:00 - 21:00
        ];
        
        foreach ($peakPeriods as $period) {
            if ($currentTime >= $period['start'] && $currentTime <= $period['end']) {
                return true;
            }
        }
        
        return false;
    }
}

if (!function_exists('get_setting')) {
    /**
     * Get a system setting value.
     */
    function get_setting(string $key, mixed $default = null): mixed
    {
        return Cache::remember("setting.{$key}", 3600, function () use ($key, $default) {
            $setting = \App\Modules\Admin\Models\Setting::where('key', $key)->first();
            
            if (!$setting) {
                return $default;
            }
            
            return match ($setting->type) {
                'integer' => (int) $setting->value,
                'boolean' => filter_var($setting->value, FILTER_VALIDATE_BOOLEAN),
                'json', 'array' => json_decode($setting->value, true),
                default => $setting->value,
            };
        });
    }
}

if (!function_exists('clear_setting_cache')) {
    /**
     * Clear cached setting.
     */
    function clear_setting_cache(string $key = null): void
    {
        if ($key) {
            Cache::forget("setting.{$key}");
        } else {
            // Clear all settings cache
            $settings = \App\Modules\Admin\Models\Setting::pluck('key');
            foreach ($settings as $settingKey) {
                Cache::forget("setting.{$settingKey}");
            }
        }
    }
}

if (!function_exists('mask_phone')) {
    /**
     * Mask phone number for privacy.
     */
    function mask_phone(string $phone): string
    {
        $phone = preg_replace('/[^0-9]/', '', $phone);
        $length = strlen($phone);
        
        if ($length <= 4) {
            return $phone;
        }
        
        $visible = 4;
        $masked = str_repeat('*', $length - $visible);
        
        return $masked . substr($phone, -$visible);
    }
}

if (!function_exists('mask_email')) {
    /**
     * Mask email for privacy.
     */
    function mask_email(string $email): string
    {
        $parts = explode('@', $email);
        
        if (count($parts) !== 2) {
            return $email;
        }
        
        $name = $parts[0];
        $domain = $parts[1];
        
        $visibleChars = min(3, strlen($name));
        $masked = substr($name, 0, $visibleChars) . str_repeat('*', max(0, strlen($name) - $visibleChars));
        
        return $masked . '@' . $domain;
    }
}

if (!function_exists('generate_reference')) {
    /**
     * Generate unique reference code.
     */
    function generate_reference(string $prefix = 'REF'): string
    {
        return $prefix . '_' . now()->format('YmdHis') . '_' . strtoupper(bin2hex(random_bytes(4)));
    }
}

if (!function_exists('format_duration')) {
    /**
     * Format duration in minutes to human readable.
     */
    function format_duration(int $minutes): string
    {
        if ($minutes < 60) {
            return "{$minutes} mins";
        }
        
        $hours = floor($minutes / 60);
        $mins = $minutes % 60;
        
        if ($mins === 0) {
            return $hours === 1 ? '1 hour' : "{$hours} hours";
        }
        
        return "{$hours}h {$mins}m";
    }
}

if (!function_exists('get_greeting')) {
    /**
     * Get time-based greeting.
     */
    function get_greeting(): string
    {
        $hour = (int) now()->format('H');
        
        if ($hour >= 5 && $hour < 12) {
            return 'Good morning';
        }
        
        if ($hour >= 12 && $hour < 17) {
            return 'Good afternoon';
        }
        
        if ($hour >= 17 && $hour < 21) {
            return 'Good evening';
        }
        
        return 'Good night';
    }
}

if (!function_exists('is_campus_location')) {
    /**
     * Check if an address is within a campus.
     */
    function is_campus_location(?\App\Modules\Location\Models\Address $address): bool
    {
        return $address && $address->campus_id !== null;
    }
}

if (!function_exists('format_rating')) {
    /**
     * Format rating for display.
     */
    function format_rating(float $rating): string
    {
        return number_format($rating, 1);
    }
}

if (!function_exists('truncate_text')) {
    /**
     * Truncate text with ellipsis.
     */
    function truncate_text(string $text, int $length = 100, string $suffix = '...'): string
    {
        if (strlen($text) <= $length) {
            return $text;
        }
        
        return rtrim(substr($text, 0, $length)) . $suffix;
    }
}
