<?php

namespace App\Modules\Auth\Controllers;

use App\Http\Controllers\Controller;
use App\Modules\Auth\Requests\RegisterRequest;
use App\Modules\Auth\Requests\LoginRequest;
use App\Modules\Auth\Requests\OtpLoginRequest;
use App\Modules\Auth\Requests\VerifyOtpRequest;
use App\Modules\Auth\Requests\ResendOtpRequest;
use App\Modules\Auth\Requests\ForgotPasswordRequest;
use App\Modules\Auth\Requests\ResetPasswordRequest;
use App\Modules\Auth\Requests\ChangePasswordRequest;
use App\Modules\Auth\Services\AuthService;
use App\Modules\Auth\Resources\UserResource;
use App\Modules\Auth\Models\OtpVerification;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class AuthController extends Controller
{
    public function __construct(
        protected AuthService $authService
    ) {}

    /**
     * Register a new customer.
     */
    public function register(RegisterRequest $request): JsonResponse
    {
        $result = $this->authService->register($request->validated());

        return response()->json([
            'success' => true,
            'message' => $result['message'],
            'data' => [
                'user' => new UserResource($result['user']),
                'otp_expires_at' => $result['otp_expires_at']->toIso8601String(),
            ],
        ], 201);
    }

    /**
     * Login with phone and password.
     */
    public function login(LoginRequest $request): JsonResponse
    {
        $result = $this->authService->login(
            $request->validated(),
            $request->ip()
        );

        if (isset($result['requires_verification']) && $result['requires_verification']) {
            return response()->json([
                'success' => true,
                'requires_verification' => true,
                'message' => $result['message'],
                'data' => [
                    'otp_expires_at' => $result['otp_expires_at']->toIso8601String(),
                ],
            ]);
        }

        return response()->json([
            'success' => true,
            'message' => 'Login successful.',
            'data' => [
                'user' => new UserResource($result['user']),
                'token' => $result['token'],
                'token_type' => $result['token_type'],
            ],
        ]);
    }

    /**
     * Request OTP login (passwordless).
     */
    public function requestOtpLogin(OtpLoginRequest $request): JsonResponse
    {
        $result = $this->authService->loginWithOtp($request->phone);

        return response()->json([
            'success' => true,
            'message' => $result['message'],
            'data' => [
                'otp_expires_at' => $result['otp_expires_at']->toIso8601String(),
            ],
        ]);
    }

    /**
     * Verify OTP.
     */
    public function verifyOtp(VerifyOtpRequest $request): JsonResponse
    {
        $result = $this->authService->verifyOtp(
            $request->phone,
            $request->otp,
            $request->purpose ?? OtpVerification::PURPOSE_LOGIN,
            $request->ip()
        );

        return response()->json([
            'success' => true,
            'message' => 'Verification successful.',
            'data' => [
                'user' => new UserResource($result['user']),
                'token' => $result['token'],
                'token_type' => $result['token_type'],
            ],
        ]);
    }

    /**
     * Resend OTP.
     */
    public function resendOtp(ResendOtpRequest $request): JsonResponse
    {
        $result = $this->authService->resendOtp(
            $request->phone,
            $request->purpose ?? OtpVerification::PURPOSE_LOGIN
        );

        return response()->json([
            'success' => true,
            'message' => $result['message'],
            'data' => [
                'otp_expires_at' => $result['otp_expires_at']->toIso8601String(),
            ],
        ]);
    }

    /**
     * Get authenticated user.
     */
    public function user(Request $request): JsonResponse
    {
        $user = $request->user()->load(['customer', 'wallet', 'addresses']);

        return response()->json([
            'success' => true,
            'data' => [
                'user' => new UserResource($user),
            ],
        ]);
    }

    /**
     * Update profile.
     */
    public function updateProfile(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'first_name' => 'sometimes|string|max:100',
            'last_name' => 'sometimes|string|max:100',
            'email' => 'sometimes|email|unique:users,email,' . $request->user()->id,
            'avatar' => 'sometimes|image|max:2048',
        ]);

        $user = $request->user();

        if ($request->hasFile('avatar')) {
            // Handle avatar upload
            $path = $request->file('avatar')->store('avatars', 'public');
            $validated['avatar'] = $path;
        }

        $user->update($validated);

        return response()->json([
            'success' => true,
            'message' => 'Profile updated successfully.',
            'data' => [
                'user' => new UserResource($user->fresh()),
            ],
        ]);
    }

    /**
     * Request password reset.
     */
    public function forgotPassword(ForgotPasswordRequest $request): JsonResponse
    {
        $result = $this->authService->requestPasswordReset($request->phone);

        return response()->json([
            'success' => true,
            'message' => $result['message'],
            'data' => isset($result['otp_expires_at']) ? [
                'otp_expires_at' => $result['otp_expires_at']->toIso8601String(),
            ] : null,
        ]);
    }

    /**
     * Reset password with OTP.
     */
    public function resetPassword(ResetPasswordRequest $request): JsonResponse
    {
        $result = $this->authService->resetPassword(
            $request->phone,
            $request->otp,
            $request->password
        );

        return response()->json([
            'success' => true,
            'message' => $result['message'],
        ]);
    }

    /**
     * Change password.
     */
    public function changePassword(ChangePasswordRequest $request): JsonResponse
    {
        $result = $this->authService->changePassword(
            $request->user(),
            $request->current_password,
            $request->password
        );

        return response()->json([
            'success' => true,
            'message' => $result['message'],
        ]);
    }

    /**
     * Logout.
     */
    public function logout(Request $request): JsonResponse
    {
        $this->authService->logout($request->user());

        return response()->json([
            'success' => true,
            'message' => 'Logged out successfully.',
        ]);
    }

    /**
     * Logout from all devices.
     */
    public function logoutAll(Request $request): JsonResponse
    {
        $this->authService->logoutAll($request->user());

        return response()->json([
            'success' => true,
            'message' => 'Logged out from all devices.',
        ]);
    }

    /**
     * Update device token for push notifications.
     */
    public function updateDeviceToken(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'device_token' => 'required|string|max:500',
        ]);

        $request->user()->update([
            'device_token' => $validated['device_token'],
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Device token updated.',
        ]);
    }
}
