<?php

namespace App\Modules\Auth\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class OtpVerification extends Model
{
    use HasFactory;

    /**
     * OTP purposes
     */
    public const PURPOSE_REGISTRATION = 'registration';
    public const PURPOSE_LOGIN = 'login';
    public const PURPOSE_PASSWORD_RESET = 'password_reset';
    public const PURPOSE_PHONE_CHANGE = 'phone_change';

    /**
     * Identifier types
     */
    public const TYPE_PHONE = 'phone';
    public const TYPE_EMAIL = 'email';

    /**
     * Maximum attempts before lockout
     */
    public const MAX_ATTEMPTS = 5;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'identifier',
        'identifier_type',
        'otp_code',
        'purpose',
        'expires_at',
        'verified_at',
        'attempts',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'expires_at' => 'datetime',
        'verified_at' => 'datetime',
    ];

    /**
     * Generate a new OTP code.
     */
    public static function generateCode(int $length = 6): string
    {
        $length = config('auth.otp_length', $length);
        return str_pad((string) random_int(0, pow(10, $length) - 1), $length, '0', STR_PAD_LEFT);
    }

    /**
     * Create a new OTP for the given identifier.
     */
    public static function createFor(
        string $identifier,
        string $purpose,
        string $identifierType = self::TYPE_PHONE
    ): self {
        // Invalidate any existing OTPs for this identifier and purpose
        self::where('identifier', $identifier)
            ->where('purpose', $purpose)
            ->whereNull('verified_at')
            ->delete();

        return self::create([
            'identifier' => $identifier,
            'identifier_type' => $identifierType,
            'otp_code' => self::generateCode(),
            'purpose' => $purpose,
            'expires_at' => now()->addMinutes(config('auth.otp_expiry_minutes', 10)),
            'attempts' => 0,
        ]);
    }

    /**
     * Check if OTP is expired.
     */
    public function isExpired(): bool
    {
        return $this->expires_at->isPast();
    }

    /**
     * Check if OTP is already verified.
     */
    public function isVerified(): bool
    {
        return !is_null($this->verified_at);
    }

    /**
     * Check if max attempts exceeded.
     */
    public function hasExceededMaxAttempts(): bool
    {
        return $this->attempts >= self::MAX_ATTEMPTS;
    }

    /**
     * Check if OTP is valid.
     */
    public function isValid(): bool
    {
        return !$this->isExpired() && !$this->isVerified() && !$this->hasExceededMaxAttempts();
    }

    /**
     * Verify the OTP code.
     */
    public function verify(string $code): bool
    {
        if (!$this->isValid()) {
            return false;
        }

        if ($this->otp_code !== $code) {
            $this->increment('attempts');
            return false;
        }

        $this->update(['verified_at' => now()]);
        return true;
    }

    /**
     * Scope for pending (unverified) OTPs.
     */
    public function scopePending($query)
    {
        return $query->whereNull('verified_at');
    }

    /**
     * Scope for valid (not expired, not verified, attempts not exceeded) OTPs.
     */
    public function scopeValid($query)
    {
        return $query->pending()
            ->where('expires_at', '>', now())
            ->where('attempts', '<', self::MAX_ATTEMPTS);
    }

    /**
     * Find valid OTP for identifier and purpose.
     */
    public static function findValidFor(string $identifier, string $purpose): ?self
    {
        return self::where('identifier', $identifier)
            ->where('purpose', $purpose)
            ->valid()
            ->latest()
            ->first();
    }

    /**
     * Check if can resend OTP (rate limiting).
     */
    public static function canResend(string $identifier, string $purpose): bool
    {
        $lastOtp = self::where('identifier', $identifier)
            ->where('purpose', $purpose)
            ->latest()
            ->first();

        if (!$lastOtp) {
            return true;
        }

        $resendInterval = config('auth.otp_resend_interval', 60);
        return $lastOtp->created_at->addSeconds($resendInterval)->isPast();
    }

    /**
     * Get seconds until can resend.
     */
    public static function secondsUntilCanResend(string $identifier, string $purpose): int
    {
        $lastOtp = self::where('identifier', $identifier)
            ->where('purpose', $purpose)
            ->latest()
            ->first();

        if (!$lastOtp) {
            return 0;
        }

        $resendInterval = config('auth.otp_resend_interval', 60);
        $canResendAt = $lastOtp->created_at->addSeconds($resendInterval);

        return max(0, now()->diffInSeconds($canResendAt, false));
    }
}
