<?php

namespace App\Modules\Auth\Requests;

use Illuminate\Foundation\Http\FormRequest;

class RegisterRequest extends FormRequest
{
    public function authorize(): bool
    {
        return true;
    }

    public function rules(): array
    {
        return [
            'first_name' => ['required', 'string', 'max:100'],
            'last_name' => ['required', 'string', 'max:100'],
            'phone' => ['required', 'string', 'regex:/^(\+234|0)[789][01]\d{8}$/', 'unique:users,phone'],
            'email' => ['nullable', 'email', 'max:255', 'unique:users,email'],
            'password' => ['required', 'string', 'min:8', 'confirmed'],
            'referral_code' => ['nullable', 'string', 'max:20'],
        ];
    }

    public function messages(): array
    {
        return [
            'phone.regex' => 'Please enter a valid Nigerian phone number.',
            'phone.unique' => 'This phone number is already registered.',
            'email.unique' => 'This email is already registered.',
        ];
    }

    protected function prepareForValidation(): void
    {
        if ($this->phone) {
            // Normalize phone number to +234 format
            $phone = preg_replace('/[^0-9]/', '', $this->phone);
            if (str_starts_with($phone, '0')) {
                $phone = '+234' . substr($phone, 1);
            } elseif (!str_starts_with($phone, '234')) {
                $phone = '+234' . $phone;
            } else {
                $phone = '+' . $phone;
            }
            $this->merge(['phone' => $phone]);
        }
    }
}
