<?php

namespace App\Modules\Auth\Services;

use App\Modules\Auth\Models\OtpVerification;
use App\Modules\Auth\Events\UserRegistered;
use App\Modules\Auth\Events\UserLoggedIn;
use App\Modules\Auth\Events\OtpSent;
use App\Modules\Auth\Events\OtpVerified;
use App\Modules\Auth\Jobs\SendOtpJob;
use App\Modules\User\Models\User;
use App\Modules\Customer\Models\Customer;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;
use Illuminate\Validation\ValidationException;

class AuthService
{
    /**
     * Register a new user.
     */
    public function register(array $data): array
    {
        return DB::transaction(function () use ($data) {
            // Check if phone already exists
            if (User::where('phone', $data['phone'])->exists()) {
                throw ValidationException::withMessages([
                    'phone' => ['This phone number is already registered.'],
                ]);
            }

            // Check if email already exists (if provided)
            if (!empty($data['email']) && User::where('email', $data['email'])->exists()) {
                throw ValidationException::withMessages([
                    'email' => ['This email is already registered.'],
                ]);
            }

            // Create user
            $user = User::create([
                'uuid' => Str::uuid()->toString(),
                'first_name' => $data['first_name'],
                'last_name' => $data['last_name'],
                'phone' => $data['phone'],
                'email' => $data['email'] ?? null,
                'password' => Hash::make($data['password']),
                'user_type' => User::TYPE_CUSTOMER,
                'status' => User::STATUS_PENDING,
            ]);

            // Create customer profile
            Customer::create([
                'user_id' => $user->id,
                'referral_code' => $this->generateReferralCode(),
                'referred_by' => $this->findReferrer($data['referral_code'] ?? null),
            ]);

            // Create wallet
            $user->wallet()->create([
                'balance' => 0,
                'pending_balance' => 0,
            ]);

            // Send OTP for phone verification
            $otp = $this->sendOtp($data['phone'], OtpVerification::PURPOSE_REGISTRATION);

            event(new UserRegistered($user));

            return [
                'user' => $user,
                'otp_expires_at' => $otp->expires_at,
                'message' => 'Registration successful. Please verify your phone number.',
            ];
        });
    }

    /**
     * Login user with phone and password.
     */
    public function login(array $credentials, string $ip): array
    {
        $user = User::where('phone', $credentials['phone'])->first();

        if (!$user || !Hash::check($credentials['password'], $user->password)) {
            throw ValidationException::withMessages([
                'phone' => ['The provided credentials are incorrect.'],
            ]);
        }

        if ($user->status === User::STATUS_SUSPENDED) {
            throw ValidationException::withMessages([
                'phone' => ['Your account has been suspended. Please contact support.'],
            ]);
        }

        if (!$user->hasVerifiedPhone()) {
            // Send OTP for verification
            $otp = $this->sendOtp($user->phone, OtpVerification::PURPOSE_LOGIN);
            
            return [
                'requires_verification' => true,
                'otp_expires_at' => $otp->expires_at,
                'message' => 'Please verify your phone number to continue.',
            ];
        }

        // Generate token
        $token = $user->createToken('auth-token', $this->getAbilitiesForUser($user))->plainTextToken;

        // Update last login
        $user->updateLastLogin($ip);

        event(new UserLoggedIn($user));

        return [
            'user' => $user->load(['customer', 'wallet']),
            'token' => $token,
            'token_type' => 'Bearer',
        ];
    }

    /**
     * Login with OTP (passwordless).
     */
    public function loginWithOtp(string $phone): array
    {
        $user = User::where('phone', $phone)->first();

        if (!$user) {
            throw ValidationException::withMessages([
                'phone' => ['No account found with this phone number.'],
            ]);
        }

        if ($user->status === User::STATUS_SUSPENDED) {
            throw ValidationException::withMessages([
                'phone' => ['Your account has been suspended.'],
            ]);
        }

        $otp = $this->sendOtp($phone, OtpVerification::PURPOSE_LOGIN);

        return [
            'otp_expires_at' => $otp->expires_at,
            'message' => 'OTP sent successfully.',
        ];
    }

    /**
     * Verify OTP and complete authentication.
     */
    public function verifyOtp(string $phone, string $code, string $purpose, string $ip): array
    {
        $otpRecord = OtpVerification::findValidFor($phone, $purpose);

        if (!$otpRecord) {
            throw ValidationException::withMessages([
                'otp' => ['OTP has expired or is invalid. Please request a new one.'],
            ]);
        }

        if (!$otpRecord->verify($code)) {
            $attemptsLeft = OtpVerification::MAX_ATTEMPTS - $otpRecord->attempts;
            throw ValidationException::withMessages([
                'otp' => ["Invalid OTP. {$attemptsLeft} attempts remaining."],
            ]);
        }

        $user = User::where('phone', $phone)->first();

        if (!$user) {
            throw ValidationException::withMessages([
                'phone' => ['User not found.'],
            ]);
        }

        // Mark phone as verified
        if (!$user->hasVerifiedPhone()) {
            $user->markPhoneAsVerified();
        }

        // Generate token
        $token = $user->createToken('auth-token', $this->getAbilitiesForUser($user))->plainTextToken;

        // Update last login
        $user->updateLastLogin($ip);

        event(new OtpVerified($user, $purpose));

        return [
            'user' => $user->load(['customer', 'wallet']),
            'token' => $token,
            'token_type' => 'Bearer',
        ];
    }

    /**
     * Send OTP to phone number.
     */
    public function sendOtp(string $phone, string $purpose): OtpVerification
    {
        // Check rate limit
        if (!OtpVerification::canResend($phone, $purpose)) {
            $seconds = OtpVerification::secondsUntilCanResend($phone, $purpose);
            throw ValidationException::withMessages([
                'phone' => ["Please wait {$seconds} seconds before requesting a new OTP."],
            ]);
        }

        $otp = OtpVerification::createFor($phone, $purpose);

        // Dispatch job to send OTP via SMS
        SendOtpJob::dispatch($phone, $otp->otp_code, $purpose);

        event(new OtpSent($phone, $purpose));

        return $otp;
    }

    /**
     * Resend OTP.
     */
    public function resendOtp(string $phone, string $purpose): array
    {
        $otp = $this->sendOtp($phone, $purpose);

        return [
            'otp_expires_at' => $otp->expires_at,
            'message' => 'OTP sent successfully.',
        ];
    }

    /**
     * Logout user.
     */
    public function logout(User $user): void
    {
        $user->currentAccessToken()->delete();
    }

    /**
     * Logout from all devices.
     */
    public function logoutAll(User $user): void
    {
        $user->tokens()->delete();
    }

    /**
     * Request password reset.
     */
    public function requestPasswordReset(string $phone): array
    {
        $user = User::where('phone', $phone)->first();

        if (!$user) {
            // Don't reveal that user doesn't exist
            return [
                'message' => 'If an account exists with this phone number, you will receive an OTP.',
            ];
        }

        $otp = $this->sendOtp($phone, OtpVerification::PURPOSE_PASSWORD_RESET);

        return [
            'otp_expires_at' => $otp->expires_at,
            'message' => 'OTP sent for password reset.',
        ];
    }

    /**
     * Reset password with OTP.
     */
    public function resetPassword(string $phone, string $code, string $newPassword): array
    {
        $otpRecord = OtpVerification::findValidFor($phone, OtpVerification::PURPOSE_PASSWORD_RESET);

        if (!$otpRecord || !$otpRecord->verify($code)) {
            throw ValidationException::withMessages([
                'otp' => ['Invalid or expired OTP.'],
            ]);
        }

        $user = User::where('phone', $phone)->first();

        if (!$user) {
            throw ValidationException::withMessages([
                'phone' => ['User not found.'],
            ]);
        }

        $user->update([
            'password' => Hash::make($newPassword),
        ]);

        // Revoke all tokens
        $user->tokens()->delete();

        return [
            'message' => 'Password reset successfully. Please login with your new password.',
        ];
    }

    /**
     * Change password.
     */
    public function changePassword(User $user, string $currentPassword, string $newPassword): array
    {
        if (!Hash::check($currentPassword, $user->password)) {
            throw ValidationException::withMessages([
                'current_password' => ['The current password is incorrect.'],
            ]);
        }

        $user->update([
            'password' => Hash::make($newPassword),
        ]);

        return [
            'message' => 'Password changed successfully.',
        ];
    }

    /**
     * Generate unique referral code.
     */
    protected function generateReferralCode(): string
    {
        do {
            $code = strtoupper(Str::random(8));
        } while (Customer::where('referral_code', $code)->exists());

        return $code;
    }

    /**
     * Find referrer by referral code.
     */
    protected function findReferrer(?string $code): ?int
    {
        if (!$code) {
            return null;
        }

        $customer = Customer::where('referral_code', strtoupper($code))->first();
        return $customer?->id;
    }

    /**
     * Get token abilities based on user type.
     */
    protected function getAbilitiesForUser(User $user): array
    {
        return match ($user->user_type) {
            User::TYPE_ADMIN => ['*'],
            User::TYPE_VENDOR => ['vendor:*', 'orders:*', 'menu:*', 'chat:*'],
            User::TYPE_RIDER => ['rider:*', 'deliveries:*', 'chat:*'],
            default => ['customer:*', 'orders:*', 'cart:*', 'chat:*'],
        };
    }
}
