<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        // Delivery Zones (Obantoko, Camp, Osiele, Odeda)
        Schema::create('zones', function (Blueprint $table) {
            $table->id();
            $table->uuid('uuid')->unique();
            $table->string('name'); // e.g., "Obantoko", "Camp/FUNAAB Gate"
            $table->string('slug')->unique();
            $table->text('description')->nullable();
            $table->decimal('base_delivery_fee', 10, 2)->default(0);
            $table->decimal('per_km_fee', 10, 2)->default(0);
            $table->decimal('minimum_order', 10, 2)->default(0);
            $table->boolean('is_active')->default(true);
            $table->json('boundaries')->nullable(); // GeoJSON polygon
            $table->decimal('center_lat', 10, 8)->nullable();
            $table->decimal('center_lng', 11, 8)->nullable();
            $table->integer('sort_order')->default(0);
            $table->timestamps();
            $table->softDeletes();
        });

        // Areas within zones
        Schema::create('areas', function (Blueprint $table) {
            $table->id();
            $table->uuid('uuid')->unique();
            $table->foreignId('zone_id')->constrained()->cascadeOnDelete();
            $table->string('name'); // e.g., "Behind NNPC", "Moshood Abiola Way"
            $table->string('slug');
            $table->decimal('additional_fee', 10, 2)->default(0);
            $table->boolean('is_active')->default(true);
            $table->decimal('latitude', 10, 8)->nullable();
            $table->decimal('longitude', 11, 8)->nullable();
            $table->timestamps();
            $table->softDeletes();

            $table->unique(['zone_id', 'slug']);
        });

        // Campus-specific locations
        Schema::create('campuses', function (Blueprint $table) {
            $table->id();
            $table->uuid('uuid')->unique();
            $table->foreignId('zone_id')->constrained()->cascadeOnDelete();
            $table->string('name'); // e.g., "FUNAAB", "FCE Osiele"
            $table->string('slug')->unique();
            $table->string('short_name')->nullable(); // e.g., "FUNAAB"
            $table->text('address')->nullable();
            $table->decimal('latitude', 10, 8)->nullable();
            $table->decimal('longitude', 11, 8)->nullable();
            $table->boolean('is_active')->default(true);
            $table->boolean('has_shuttle_delivery')->default(false);
            $table->timestamps();
            $table->softDeletes();
        });

        // Campus locations (hostels, departments, lecture theatres)
        Schema::create('campus_locations', function (Blueprint $table) {
            $table->id();
            $table->uuid('uuid')->unique();
            $table->foreignId('campus_id')->constrained()->cascadeOnDelete();
            $table->enum('type', ['hostel', 'department', 'lecture_theatre', 'gate', 'library', 'cafeteria', 'other']);
            $table->string('name'); // e.g., "Hall 1", "COLENG Building"
            $table->string('block')->nullable(); // For hostels
            $table->text('landmark')->nullable();
            $table->decimal('additional_fee', 10, 2)->default(0);
            $table->boolean('is_active')->default(true);
            $table->decimal('latitude', 10, 8)->nullable();
            $table->decimal('longitude', 11, 8)->nullable();
            $table->timestamps();
            $table->softDeletes();

            $table->index(['campus_id', 'type']);
        });

        // Delivery fee matrix (zone to zone pricing)
        Schema::create('delivery_fee_matrix', function (Blueprint $table) {
            $table->id();
            $table->foreignId('from_zone_id')->constrained('zones')->cascadeOnDelete();
            $table->foreignId('to_zone_id')->constrained('zones')->cascadeOnDelete();
            $table->decimal('base_fee', 10, 2);
            $table->decimal('per_km_fee', 10, 2)->default(0);
            $table->integer('estimated_minutes')->nullable();
            $table->boolean('is_active')->default(true);
            $table->timestamps();

            $table->unique(['from_zone_id', 'to_zone_id']);
        });
    }

    public function down(): void
    {
        Schema::dropIfExists('delivery_fee_matrix');
        Schema::dropIfExists('campus_locations');
        Schema::dropIfExists('campuses');
        Schema::dropIfExists('areas');
        Schema::dropIfExists('zones');
    }
};