<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        // User Wallets (Customers & Vendors)
        Schema::create('wallets', function (Blueprint $table) {
            $table->id();
            $table->uuid('uuid')->unique();
            $table->foreignId('user_id')->constrained()->cascadeOnDelete();
            $table->decimal('balance', 12, 2)->default(0);
            $table->decimal('pending_balance', 12, 2)->default(0);
            $table->string('currency', 3)->default('NGN');
            $table->boolean('is_active')->default(true);
            $table->string('pin')->nullable(); // Wallet PIN
            $table->timestamps();

            $table->unique('user_id');
        });

        // Wallet Transactions
        Schema::create('wallet_transactions', function (Blueprint $table) {
            $table->id();
            $table->uuid('uuid')->unique();
            $table->foreignId('wallet_id')->constrained()->cascadeOnDelete();
            $table->foreignId('user_id')->constrained()->cascadeOnDelete();
            $table->enum('type', ['credit', 'debit']);
            $table->enum('category', [
                'deposit', 'withdrawal', 'payment', 'refund', 
                'commission', 'bonus', 'transfer', 'settlement'
            ]);
            $table->decimal('amount', 12, 2);
            $table->decimal('balance_before', 12, 2);
            $table->decimal('balance_after', 12, 2);
            $table->string('reference')->unique();
            $table->text('description')->nullable();
            $table->json('metadata')->nullable();
            $table->enum('status', ['pending', 'completed', 'failed', 'reversed'])->default('pending');
            $table->timestamps();

            $table->index(['wallet_id', 'type', 'status']);
            $table->index('reference');
        });

        // Rider Wallets (Separate for commission tracking)
        Schema::create('rider_wallets', function (Blueprint $table) {
            $table->id();
            $table->uuid('uuid')->unique();
            $table->foreignId('rider_profile_id')->constrained()->cascadeOnDelete();
            $table->decimal('balance', 12, 2)->default(0);
            $table->decimal('pending_balance', 12, 2)->default(0);
            $table->decimal('total_earnings', 12, 2)->default(0);
            $table->decimal('total_withdrawn', 12, 2)->default(0);
            $table->string('currency', 3)->default('NGN');
            $table->boolean('is_active')->default(true);
            $table->timestamps();

            $table->unique('rider_profile_id');
        });

        // Rider Transactions
        Schema::create('rider_transactions', function (Blueprint $table) {
            $table->id();
            $table->uuid('uuid')->unique();
            $table->foreignId('rider_wallet_id')->constrained()->cascadeOnDelete();
            $table->foreignId('order_id')->nullable();
            $table->enum('type', ['credit', 'debit']);
            $table->enum('category', [
                'delivery_fee', 'tip', 'bonus', 'penalty',
                'withdrawal', 'adjustment', 'commission'
            ]);
            $table->decimal('amount', 12, 2);
            $table->decimal('balance_before', 12, 2);
            $table->decimal('balance_after', 12, 2);
            $table->string('reference')->unique();
            $table->text('description')->nullable();
            $table->json('metadata')->nullable();
            $table->enum('status', ['pending', 'completed', 'failed', 'reversed'])->default('pending');
            $table->timestamps();

            $table->index(['rider_wallet_id', 'type', 'status']);
        });
    }

    public function down(): void
    {
        Schema::dropIfExists('rider_transactions');
        Schema::dropIfExists('rider_wallets');
        Schema::dropIfExists('wallet_transactions');
        Schema::dropIfExists('wallets');
    }
};