<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        Schema::create('restaurants', function (Blueprint $table) {
            $table->id();
            $table->uuid('uuid')->unique();
            $table->foreignId('user_id')->constrained()->cascadeOnDelete(); // Vendor owner
            $table->foreignId('zone_id')->constrained()->cascadeOnDelete();
            
            // Basic Info
            $table->string('name');
            $table->string('slug')->unique();
            $table->text('description')->nullable();
            $table->string('logo')->nullable();
            $table->string('banner')->nullable();
            $table->string('phone');
            $table->string('email')->nullable();
            $table->text('address');
            $table->decimal('latitude', 10, 8)->nullable();
            $table->decimal('longitude', 11, 8)->nullable();
            
            // Categories/Tags
            $table->json('cuisine_types')->nullable(); // ["Nigerian", "Fast Food", "Chinese"]
            $table->json('tags')->nullable(); // ["Halal", "Vegetarian Options", "24/7"]
            
            // Business Settings
            $table->decimal('minimum_order', 10, 2)->default(0);
            $table->integer('preparation_time')->default(30); // Average minutes
            $table->decimal('delivery_fee', 10, 2)->nullable(); // If restaurant handles own delivery
            $table->boolean('free_delivery')->default(false);
            $table->decimal('free_delivery_threshold', 10, 2)->nullable();
            
            // Ratings
            $table->decimal('rating', 3, 2)->default(0);
            $table->integer('rating_count')->default(0);
            $table->integer('total_orders')->default(0);
            
            // Status
            $table->enum('status', ['pending', 'active', 'inactive', 'suspended', 'closed'])->default('pending');
            $table->boolean('is_featured')->default(false);
            $table->boolean('is_verified')->default(false);
            $table->timestamp('verified_at')->nullable();
            $table->boolean('is_open')->default(false);
            $table->boolean('accepts_preorders')->default(true);
            $table->boolean('has_own_riders')->default(false);
            
            // Commission
            $table->decimal('commission_rate', 5, 2)->nullable(); // Override vendor default
            
            // Operating Hours stored separately
            $table->timestamps();
            $table->softDeletes();

            $table->index(['zone_id', 'status', 'is_open']);
            $table->index(['status', 'is_featured']);
            $table->fullText(['name', 'description']);
        });

        // Restaurant Branches
        Schema::create('restaurant_branches', function (Blueprint $table) {
            $table->id();
            $table->uuid('uuid')->unique();
            $table->foreignId('restaurant_id')->constrained()->cascadeOnDelete();
            $table->foreignId('zone_id')->constrained()->cascadeOnDelete();
            $table->string('name');
            $table->string('slug');
            $table->string('phone')->nullable();
            $table->text('address');
            $table->decimal('latitude', 10, 8)->nullable();
            $table->decimal('longitude', 11, 8)->nullable();
            $table->boolean('is_active')->default(true);
            $table->boolean('is_open')->default(false);
            $table->timestamps();
            $table->softDeletes();

            $table->unique(['restaurant_id', 'slug']);
        });

        // Operating Hours
        Schema::create('restaurant_operating_hours', function (Blueprint $table) {
            $table->id();
            $table->foreignId('restaurant_id')->constrained()->cascadeOnDelete();
            $table->foreignId('branch_id')->nullable()->constrained('restaurant_branches')->cascadeOnDelete();
            $table->enum('day', ['monday', 'tuesday', 'wednesday', 'thursday', 'friday', 'saturday', 'sunday']);
            $table->time('open_time')->nullable();
            $table->time('close_time')->nullable();
            $table->boolean('is_closed')->default(false);
            $table->boolean('is_24_hours')->default(false);
            $table->timestamps();

            $table->unique(['restaurant_id', 'branch_id', 'day']);
        });

        // Peak Hours (for surge pricing or extended prep times)
        Schema::create('restaurant_peak_hours', function (Blueprint $table) {
            $table->id();
            $table->foreignId('restaurant_id')->constrained()->cascadeOnDelete();
            $table->enum('day', ['monday', 'tuesday', 'wednesday', 'thursday', 'friday', 'saturday', 'sunday']);
            $table->time('start_time');
            $table->time('end_time');
            $table->integer('additional_prep_time')->default(0); // Extra minutes
            $table->decimal('surge_multiplier', 4, 2)->default(1.00);
            $table->boolean('is_active')->default(true);
            $table->timestamps();
        });
    }

    public function down(): void
    {
        Schema::dropIfExists('restaurant_peak_hours');
        Schema::dropIfExists('restaurant_operating_hours');
        Schema::dropIfExists('restaurant_branches');
        Schema::dropIfExists('restaurants');
    }
};