<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        Schema::create('orders', function (Blueprint $table) {
            $table->id();
            $table->uuid('uuid')->unique();
            $table->string('order_number')->unique();
            
            // Relationships
            $table->foreignId('user_id')->constrained()->cascadeOnDelete();
            $table->foreignId('restaurant_id')->constrained()->cascadeOnDelete();
            $table->foreignId('branch_id')->nullable()->constrained('restaurant_branches')->nullOnDelete();
            $table->foreignId('rider_profile_id')->nullable()->constrained()->nullOnDelete();
            $table->foreignId('delivery_address_id')->nullable()->constrained('user_addresses')->nullOnDelete();
            
            // Order Type
            $table->enum('type', ['delivery', 'pickup', 'dine_in'])->default('delivery');
            $table->enum('delivery_type', ['regular', 'campus', 'shuttle'])->default('regular');
            
            // Amounts
            $table->decimal('subtotal', 12, 2);
            $table->decimal('delivery_fee', 10, 2)->default(0);
            $table->decimal('service_fee', 10, 2)->default(0);
            $table->decimal('packaging_fee', 10, 2)->default(0);
            $table->decimal('tip', 10, 2)->default(0);
            $table->decimal('discount', 10, 2)->default(0);
            $table->decimal('tax', 10, 2)->default(0);
            $table->decimal('total', 12, 2);
            
            // Platform Commission
            $table->decimal('commission_rate', 5, 2)->default(0);
            $table->decimal('commission_amount', 10, 2)->default(0);
            $table->decimal('vendor_earnings', 12, 2)->default(0);
            $table->decimal('rider_earnings', 10, 2)->default(0);
            
            // Status
            $table->enum('status', [
                'pending',           // Just placed
                'confirmed',         // Restaurant confirmed
                'preparing',         // Being prepared
                'ready',             // Ready for pickup
                'rider_assigned',    // Rider accepted
                'picked_up',         // Rider picked up
                'on_the_way',        // In transit
                'arrived',           // Rider at location
                'delivered',         // Completed
                'cancelled',         // Cancelled
                'refunded'           // Refunded
            ])->default('pending');
            
            // Payment
            $table->enum('payment_method', ['wallet', 'card', 'bank_transfer', 'cash', 'paystack'])->default('cash');
            $table->enum('payment_status', ['pending', 'paid', 'failed', 'refunded', 'partial_refund'])->default('pending');
            $table->timestamp('paid_at')->nullable();
            
            // Delivery Info
            $table->json('delivery_address_snapshot')->nullable(); // Store address at time of order
            $table->text('delivery_instructions')->nullable();
            $table->integer('estimated_prep_time')->nullable(); // Minutes
            $table->integer('estimated_delivery_time')->nullable(); // Minutes
            $table->decimal('delivery_distance', 8, 2)->nullable(); // KM
            
            // Timestamps
            $table->timestamp('confirmed_at')->nullable();
            $table->timestamp('preparing_at')->nullable();
            $table->timestamp('ready_at')->nullable();
            $table->timestamp('picked_up_at')->nullable();
            $table->timestamp('delivered_at')->nullable();
            $table->timestamp('cancelled_at')->nullable();
            $table->timestamp('scheduled_for')->nullable(); // For pre-orders
            
            // Cancellation
            $table->string('cancelled_by')->nullable(); // user, vendor, rider, admin, system
            $table->text('cancellation_reason')->nullable();
            
            // Promo/Coupon
            $table->string('coupon_code')->nullable();
            $table->foreignId('coupon_id')->nullable();
            
            // Meta
            $table->text('special_instructions')->nullable();
            $table->json('metadata')->nullable();
            $table->boolean('is_preorder')->default(false);
            $table->boolean('is_rated')->default(false);
            
            $table->timestamps();
            $table->softDeletes();

            $table->index(['user_id', 'status']);
            $table->index(['restaurant_id', 'status']);
            $table->index(['rider_profile_id', 'status']);
            $table->index(['status', 'created_at']);
            $table->index('order_number');
        });

        // Order Items
        Schema::create('order_items', function (Blueprint $table) {
            $table->id();
            $table->uuid('uuid')->unique();
            $table->foreignId('order_id')->constrained()->cascadeOnDelete();
            $table->foreignId('menu_item_id')->constrained()->cascadeOnDelete();
            $table->foreignId('item_variant_id')->nullable()->constrained()->nullOnDelete();
            
            // Snapshot of item at order time
            $table->string('item_name');
            $table->string('variant_name')->nullable();
            $table->text('item_description')->nullable();
            $table->string('item_image')->nullable();
            
            $table->integer('quantity');
            $table->decimal('unit_price', 10, 2);
            $table->decimal('total_price', 10, 2);
            
            // Addons
            $table->json('selected_addons')->nullable();
            $table->decimal('addons_total', 10, 2)->default(0);
            
            $table->text('special_instructions')->nullable();
            $table->timestamps();

            $table->index('order_id');
        });

        // Order Status Logs
        Schema::create('order_status_logs', function (Blueprint $table) {
            $table->id();
            $table->foreignId('order_id')->constrained()->cascadeOnDelete();
            $table->string('status');
            $table->string('previous_status')->nullable();
            $table->string('changed_by_type')->nullable(); // user, vendor, rider, admin, system
            $table->unsignedBigInteger('changed_by_id')->nullable();
            $table->text('note')->nullable();
            $table->json('metadata')->nullable();
            $table->decimal('latitude', 10, 8)->nullable();
            $table->decimal('longitude', 11, 8)->nullable();
            $table->timestamps();

            $table->index(['order_id', 'created_at']);
        });
    }

    public function down(): void
    {
        Schema::dropIfExists('order_status_logs');
        Schema::dropIfExists('order_items');
        Schema::dropIfExists('orders');
    }
};