<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        Schema::create('payments', function (Blueprint $table) {
            $table->id();
            $table->uuid('uuid')->unique();
            $table->foreignId('order_id')->constrained()->cascadeOnDelete();
            $table->foreignId('user_id')->constrained()->cascadeOnDelete();
            
            $table->string('reference')->unique();
            $table->string('gateway')->default('paystack'); // paystack, flutterwave, wallet
            $table->string('gateway_reference')->nullable();
            $table->string('gateway_response')->nullable();
            
            $table->decimal('amount', 12, 2);
            $table->decimal('gateway_fee', 10, 2)->default(0);
            $table->string('currency', 3)->default('NGN');
            
            $table->enum('status', ['pending', 'processing', 'successful', 'failed', 'refunded'])->default('pending');
            $table->enum('payment_method', ['card', 'bank_transfer', 'ussd', 'wallet', 'cash']);
            
            $table->json('gateway_metadata')->nullable();
            $table->timestamp('paid_at')->nullable();
            $table->timestamps();

            $table->index(['order_id', 'status']);
            $table->index('reference');
        });

        // Refunds
        Schema::create('refunds', function (Blueprint $table) {
            $table->id();
            $table->uuid('uuid')->unique();
            $table->foreignId('payment_id')->constrained()->cascadeOnDelete();
            $table->foreignId('order_id')->constrained()->cascadeOnDelete();
            $table->foreignId('user_id')->constrained()->cascadeOnDelete();
            $table->foreignId('processed_by')->nullable()->constrained('users')->nullOnDelete();
            
            $table->string('reference')->unique();
            $table->decimal('amount', 12, 2);
            $table->text('reason')->nullable();
            $table->enum('status', ['pending', 'processing', 'completed', 'failed'])->default('pending');
            $table->enum('refund_to', ['wallet', 'source', 'bank'])->default('wallet');
            
            $table->string('gateway_reference')->nullable();
            $table->json('gateway_response')->nullable();
            
            $table->timestamp('processed_at')->nullable();
            $table->timestamps();

            $table->index(['order_id', 'status']);
        });

        // Settlements (Vendor payouts)
        Schema::create('settlements', function (Blueprint $table) {
            $table->id();
            $table->uuid('uuid')->unique();
            $table->foreignId('user_id')->constrained()->cascadeOnDelete(); // Vendor user
            $table->foreignId('restaurant_id')->constrained()->cascadeOnDelete();
            
            $table->string('reference')->unique();
            $table->date('period_start');
            $table->date('period_end');
            
            $table->integer('total_orders')->default(0);
            $table->decimal('gross_amount', 12, 2)->default(0);
            $table->decimal('commission_amount', 10, 2)->default(0);
            $table->decimal('delivery_fees_collected', 10, 2)->default(0);
            $table->decimal('refunds_amount', 10, 2)->default(0);
            $table->decimal('adjustments', 10, 2)->default(0);
            $table->decimal('net_amount', 12, 2)->default(0);
            
            $table->enum('status', ['pending', 'processing', 'completed', 'failed'])->default('pending');
            $table->enum('payout_method', ['bank_transfer', 'wallet'])->default('bank_transfer');
            
            $table->string('bank_name')->nullable();
            $table->string('account_number')->nullable();
            $table->string('account_name')->nullable();
            
            $table->string('gateway_reference')->nullable();
            $table->json('gateway_response')->nullable();
            
            $table->timestamp('processed_at')->nullable();
            $table->text('notes')->nullable();
            $table->timestamps();

            $table->index(['restaurant_id', 'status']);
            $table->index(['period_start', 'period_end']);
        });
    }

    public function down(): void
    {
        Schema::dropIfExists('settlements');
        Schema::dropIfExists('refunds');
        Schema::dropIfExists('payments');
    }
};