<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        // Ad Positions
        Schema::create('ad_positions', function (Blueprint $table) {
            $table->id();
            $table->string('name'); // "Top Banner", "Side Banner Left", "Category Page"
            $table->string('slug')->unique();
            $table->text('description')->nullable();
            $table->enum('type', ['banner', 'sidebar', 'inline', 'popup', 'native']);
            $table->integer('width')->nullable(); // Recommended width
            $table->integer('height')->nullable(); // Recommended height
            $table->string('page')->nullable(); // homepage, category, restaurant, checkout
            $table->decimal('price_per_day', 10, 2)->default(0);
            $table->decimal('price_per_impression', 8, 4)->default(0); // CPM
            $table->decimal('price_per_click', 8, 2)->default(0); // CPC
            $table->boolean('is_active')->default(true);
            $table->integer('sort_order')->default(0);
            $table->timestamps();
        });

        // Ads
        Schema::create('ads', function (Blueprint $table) {
            $table->id();
            $table->uuid('uuid')->unique();
            $table->foreignId('user_id')->nullable()->constrained()->nullOnDelete(); // Advertiser
            $table->foreignId('ad_position_id')->constrained()->cascadeOnDelete();
            $table->foreignId('zone_id')->nullable()->constrained()->nullOnDelete(); // Target zone
            
            $table->string('title');
            $table->text('description')->nullable();
            $table->string('image');
            $table->string('mobile_image')->nullable();
            $table->string('link_url')->nullable();
            $table->enum('link_type', ['external', 'restaurant', 'category', 'product', 'promo'])->default('external');
            $table->unsignedBigInteger('link_id')->nullable(); // If linking to internal resource
            
            // Business Info
            $table->string('business_name');
            $table->string('business_phone')->nullable();
            $table->string('business_category')->nullable(); // barber, boutique, hotel, etc.
            
            // Scheduling
            $table->date('start_date');
            $table->date('end_date');
            $table->time('daily_start_time')->nullable();
            $table->time('daily_end_time')->nullable();
            $table->json('days_of_week')->nullable(); // [1,2,3,4,5] for Mon-Fri
            
            // Targeting
            $table->json('target_zones')->nullable();
            $table->json('target_pages')->nullable();
            $table->json('target_user_types')->nullable(); // customer, vendor, rider
            
            // Budget
            $table->enum('billing_type', ['daily', 'impression', 'click', 'fixed'])->default('daily');
            $table->decimal('budget', 12, 2)->nullable();
            $table->decimal('daily_budget', 10, 2)->nullable();
            $table->decimal('spent', 12, 2)->default(0);
            
            // Stats
            $table->integer('impressions')->default(0);
            $table->integer('clicks')->default(0);
            $table->decimal('ctr', 5, 2)->default(0); // Click-through rate
            
            // Status
            $table->enum('status', ['draft', 'pending', 'active', 'paused', 'completed', 'rejected'])->default('draft');
            $table->text('rejection_reason')->nullable();
            $table->integer('priority')->default(0); // Higher = shown more
            
            $table->timestamps();
            $table->softDeletes();

            $table->index(['ad_position_id', 'status', 'start_date', 'end_date']);
            $table->index(['status', 'priority']);
        });

        // Ad Impressions Log
        Schema::create('ad_impressions', function (Blueprint $table) {
            $table->id();
            $table->foreignId('ad_id')->constrained()->cascadeOnDelete();
            $table->foreignId('user_id')->nullable()->constrained()->nullOnDelete();
            $table->string('session_id')->nullable();
            $table->string('ip_address', 45)->nullable();
            $table->string('user_agent')->nullable();
            $table->string('page')->nullable();
            $table->string('referrer')->nullable();
            $table->timestamp('created_at');

            $table->index(['ad_id', 'created_at']);
        });

        // Ad Clicks Log
        Schema::create('ad_clicks', function (Blueprint $table) {
            $table->id();
            $table->foreignId('ad_id')->constrained()->cascadeOnDelete();
            $table->foreignId('user_id')->nullable()->constrained()->nullOnDelete();
            $table->string('session_id')->nullable();
            $table->string('ip_address', 45)->nullable();
            $table->string('user_agent')->nullable();
            $table->string('page')->nullable();
            $table->string('referrer')->nullable();
            $table->timestamp('created_at');

            $table->index(['ad_id', 'created_at']);
        });

        // Ad Payments
        Schema::create('ad_payments', function (Blueprint $table) {
            $table->id();
            $table->uuid('uuid')->unique();
            $table->foreignId('ad_id')->constrained()->cascadeOnDelete();
            $table->foreignId('user_id')->constrained()->cascadeOnDelete();
            
            $table->string('reference')->unique();
            $table->decimal('amount', 12, 2);
            $table->enum('status', ['pending', 'completed', 'failed', 'refunded'])->default('pending');
            $table->string('payment_method')->nullable();
            $table->string('gateway_reference')->nullable();
            $table->json('metadata')->nullable();
            $table->timestamp('paid_at')->nullable();
            $table->timestamps();

            $table->index(['ad_id', 'status']);
        });
    }

    public function down(): void
    {
        Schema::dropIfExists('ad_payments');
        Schema::dropIfExists('ad_clicks');
        Schema::dropIfExists('ad_impressions');
        Schema::dropIfExists('ads');
        Schema::dropIfExists('ad_positions');
    }
};