/**
 * CampusEats Service Worker
 * Handles caching, offline support, and background sync
 */

const CACHE_VERSION = 'v1.0.0';
const STATIC_CACHE = `campuseats-static-${CACHE_VERSION}`;
const DYNAMIC_CACHE = `campuseats-dynamic-${CACHE_VERSION}`;
const IMAGE_CACHE = `campuseats-images-${CACHE_VERSION}`;
const API_CACHE = `campuseats-api-${CACHE_VERSION}`;

// Files to cache immediately on install
const STATIC_FILES = [
    '/',
    '/offline',
    '/manifest.json',
    '/css/app.css',
    '/js/app.js',
    '/icons/icon-192x192.png',
    '/icons/icon-512x512.png',
    '/images/logo.png',
    '/images/placeholder-food.png',
    '/images/placeholder-restaurant.png',
];

// API endpoints to cache (stale-while-revalidate)
const CACHEABLE_API_ROUTES = [
    '/api/v1/customer/public/zones',
    '/api/v1/customer/public/restaurant-categories',
    '/api/v1/customer/restaurants',
];

// Install event - cache static assets
self.addEventListener('install', (event) => {
    console.log('[SW] Installing Service Worker...');
    
    event.waitUntil(
        caches.open(STATIC_CACHE)
            .then((cache) => {
                console.log('[SW] Caching static files');
                return cache.addAll(STATIC_FILES);
            })
            .then(() => self.skipWaiting())
    );
});

// Activate event - clean up old caches
self.addEventListener('activate', (event) => {
    console.log('[SW] Activating Service Worker...');
    
    event.waitUntil(
        caches.keys()
            .then((cacheNames) => {
                return Promise.all(
                    cacheNames
                        .filter((name) => {
                            return name.startsWith('campuseats-') && 
                                   name !== STATIC_CACHE && 
                                   name !== DYNAMIC_CACHE && 
                                   name !== IMAGE_CACHE &&
                                   name !== API_CACHE;
                        })
                        .map((name) => {
                            console.log('[SW] Deleting old cache:', name);
                            return caches.delete(name);
                        })
                );
            })
            .then(() => self.clients.claim())
    );
});

// Fetch event - serve from cache or network
self.addEventListener('fetch', (event) => {
    const { request } = event;
    const url = new URL(request.url);

    // Skip non-GET requests
    if (request.method !== 'GET') {
        return;
    }

    // Handle API requests
    if (url.pathname.startsWith('/api/')) {
        event.respondWith(handleApiRequest(request));
        return;
    }

    // Handle image requests
    if (isImageRequest(request)) {
        event.respondWith(handleImageRequest(request));
        return;
    }

    // Handle static assets and pages
    event.respondWith(handleStaticRequest(request));
});

/**
 * Handle API requests with stale-while-revalidate strategy
 */
async function handleApiRequest(request) {
    const url = new URL(request.url);
    
    // Check if this is a cacheable API route
    const isCacheable = CACHEABLE_API_ROUTES.some(route => 
        url.pathname.includes(route)
    );

    if (!isCacheable) {
        // Network-only for non-cacheable APIs
        try {
            return await fetch(request);
        } catch (error) {
            return new Response(
                JSON.stringify({ error: 'Network error', offline: true }),
                { 
                    status: 503,
                    headers: { 'Content-Type': 'application/json' }
                }
            );
        }
    }

    // Stale-while-revalidate for cacheable APIs
    const cache = await caches.open(API_CACHE);
    const cachedResponse = await cache.match(request);

    const fetchPromise = fetch(request)
        .then((networkResponse) => {
            if (networkResponse.ok) {
                cache.put(request, networkResponse.clone());
            }
            return networkResponse;
        })
        .catch(() => cachedResponse);

    return cachedResponse || fetchPromise;
}

/**
 * Handle image requests with cache-first strategy
 */
async function handleImageRequest(request) {
    const cache = await caches.open(IMAGE_CACHE);
    const cachedResponse = await cache.match(request);

    if (cachedResponse) {
        return cachedResponse;
    }

    try {
        const networkResponse = await fetch(request);
        
        if (networkResponse.ok) {
            cache.put(request, networkResponse.clone());
        }
        
        return networkResponse;
    } catch (error) {
        // Return placeholder image for failed image requests
        const placeholderResponse = await cache.match('/images/placeholder-food.png');
        return placeholderResponse || new Response('', { status: 404 });
    }
}

/**
 * Handle static assets with cache-first, network fallback
 */
async function handleStaticRequest(request) {
    // Check static cache first
    const staticCache = await caches.open(STATIC_CACHE);
    const staticResponse = await staticCache.match(request);

    if (staticResponse) {
        return staticResponse;
    }

    // Check dynamic cache
    const dynamicCache = await caches.open(DYNAMIC_CACHE);
    const dynamicResponse = await dynamicCache.match(request);

    if (dynamicResponse) {
        return dynamicResponse;
    }

    // Try network
    try {
        const networkResponse = await fetch(request);
        
        // Cache successful responses
        if (networkResponse.ok && request.url.startsWith(self.location.origin)) {
            dynamicCache.put(request, networkResponse.clone());
        }
        
        return networkResponse;
    } catch (error) {
        // Return offline page for navigation requests
        if (request.mode === 'navigate') {
            const offlinePage = await staticCache.match('/offline');
            return offlinePage || new Response('Offline', { status: 503 });
        }
        
        return new Response('', { status: 404 });
    }
}

/**
 * Check if request is for an image
 */
function isImageRequest(request) {
    const url = new URL(request.url);
    const imageExtensions = ['.jpg', '.jpeg', '.png', '.gif', '.webp', '.svg', '.ico'];
    return imageExtensions.some(ext => url.pathname.toLowerCase().endsWith(ext));
}

// Background Sync for offline orders
self.addEventListener('sync', (event) => {
    console.log('[SW] Background Sync:', event.tag);
    
    if (event.tag === 'sync-orders') {
        event.waitUntil(syncQueuedOrders());
    }
    
    if (event.tag === 'sync-location') {
        event.waitUntil(syncRiderLocation());
    }
});

/**
 * Sync queued orders when back online
 */
async function syncQueuedOrders() {
    try {
        const db = await openDB();
        const orders = await getQueuedOrders(db);
        
        for (const order of orders) {
            try {
                const response = await fetch('/api/v1/customer/orders', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'Authorization': `Bearer ${order.token}`,
                    },
                    body: JSON.stringify(order.data),
                });
                
                if (response.ok) {
                    await removeQueuedOrder(db, order.id);
                    await notifyClients('order-synced', { orderId: order.id });
                }
            } catch (error) {
                console.error('[SW] Failed to sync order:', error);
            }
        }
    } catch (error) {
        console.error('[SW] Error syncing orders:', error);
    }
}

/**
 * Sync rider location updates
 */
async function syncRiderLocation() {
    try {
        const db = await openDB();
        const locations = await getQueuedLocations(db);
        
        for (const location of locations) {
            try {
                await fetch('/api/v1/rider/availability/location', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'Authorization': `Bearer ${location.token}`,
                    },
                    body: JSON.stringify(location.data),
                });
                
                await removeQueuedLocation(db, location.id);
            } catch (error) {
                console.error('[SW] Failed to sync location:', error);
            }
        }
    } catch (error) {
        console.error('[SW] Error syncing locations:', error);
    }
}

// Push Notifications
self.addEventListener('push', (event) => {
    console.log('[SW] Push received:', event);
    
    let data = { title: 'CampusEats', body: 'You have a new notification' };
    
    if (event.data) {
        try {
            data = event.data.json();
        } catch (e) {
            data.body = event.data.text();
        }
    }
    
    const options = {
        body: data.body,
        icon: '/icons/icon-192x192.png',
        badge: '/icons/badge-72x72.png',
        vibrate: [100, 50, 100],
        data: data.data || {},
        actions: data.actions || [],
        tag: data.tag || 'default',
        renotify: true,
    };
    
    event.waitUntil(
        self.registration.showNotification(data.title, options)
    );
});

// Notification click handler
self.addEventListener('notificationclick', (event) => {
    console.log('[SW] Notification clicked:', event);
    
    event.notification.close();
    
    const urlToOpen = event.notification.data?.url || '/';
    
    event.waitUntil(
        clients.matchAll({ type: 'window', includeUncontrolled: true })
            .then((clientList) => {
                // Try to focus existing window
                for (const client of clientList) {
                    if (client.url === urlToOpen && 'focus' in client) {
                        return client.focus();
                    }
                }
                // Open new window
                if (clients.openWindow) {
                    return clients.openWindow(urlToOpen);
                }
            })
    );
});

// Helper: Notify all clients
async function notifyClients(type, data) {
    const allClients = await clients.matchAll({ includeUncontrolled: true });
    allClients.forEach((client) => {
        client.postMessage({ type, data });
    });
}

// IndexedDB helpers for offline queue
function openDB() {
    return new Promise((resolve, reject) => {
        const request = indexedDB.open('CampusEatsOffline', 1);
        
        request.onerror = () => reject(request.error);
        request.onsuccess = () => resolve(request.result);
        
        request.onupgradeneeded = (event) => {
            const db = event.target.result;
            
            if (!db.objectStoreNames.contains('queuedOrders')) {
                db.createObjectStore('queuedOrders', { keyPath: 'id', autoIncrement: true });
            }
            
            if (!db.objectStoreNames.contains('queuedLocations')) {
                db.createObjectStore('queuedLocations', { keyPath: 'id', autoIncrement: true });
            }
        };
    });
}

function getQueuedOrders(db) {
    return new Promise((resolve, reject) => {
        const transaction = db.transaction('queuedOrders', 'readonly');
        const store = transaction.objectStore('queuedOrders');
        const request = store.getAll();
        
        request.onerror = () => reject(request.error);
        request.onsuccess = () => resolve(request.result);
    });
}

function removeQueuedOrder(db, id) {
    return new Promise((resolve, reject) => {
        const transaction = db.transaction('queuedOrders', 'readwrite');
        const store = transaction.objectStore('queuedOrders');
        const request = store.delete(id);
        
        request.onerror = () => reject(request.error);
        request.onsuccess = () => resolve();
    });
}

function getQueuedLocations(db) {
    return new Promise((resolve, reject) => {
        const transaction = db.transaction('queuedLocations', 'readonly');
        const store = transaction.objectStore('queuedLocations');
        const request = store.getAll();
        
        request.onerror = () => reject(request.error);
        request.onsuccess = () => resolve(request.result);
    });
}

function removeQueuedLocation(db, id) {
    return new Promise((resolve, reject) => {
        const transaction = db.transaction('queuedLocations', 'readwrite');
        const store = transaction.objectStore('queuedLocations');
        const request = store.delete(id);
        
        request.onerror = () => reject(request.error);
        request.onsuccess = () => resolve();
    });
}

console.log('[SW] Service Worker loaded');
