<?php

use Illuminate\Support\Facades\Route;

/*
|--------------------------------------------------------------------------
| Customer API Routes
|--------------------------------------------------------------------------
|
| Routes for customer-facing features.
| All routes are prefixed with /api/v1/customer
|
*/

Route::middleware(['auth:sanctum', 'user.type:customer'])->group(function () {
    // Restaurants
    Route::prefix('restaurants')->group(function () {
        Route::get('/', 'RestaurantController@index')->name('customer.restaurants.index');
        Route::get('/featured', 'RestaurantController@featured')->name('customer.restaurants.featured');
        Route::get('/nearby', 'RestaurantController@nearby')->name('customer.restaurants.nearby');
        Route::get('/categories', 'RestaurantController@categories')->name('customer.restaurants.categories');
        Route::get('/category/{slug}', 'RestaurantController@byCategory')->name('customer.restaurants.by-category');
        Route::get('/{uuid}', 'RestaurantController@show')->name('customer.restaurants.show');
        Route::get('/{uuid}/menu', 'RestaurantController@menu')->name('customer.restaurants.menu');
        Route::get('/{uuid}/reviews', 'RestaurantController@reviews')->name('customer.restaurants.reviews');
        Route::post('/{uuid}/favorite', 'RestaurantController@toggleFavorite')->name('customer.restaurants.favorite');
    });

    // Cart
    Route::prefix('cart')->group(function () {
        Route::get('/', 'CartController@show')->name('customer.cart.show');
        Route::post('/add', 'CartController@addItem')->name('customer.cart.add');
        Route::put('/item/{id}', 'CartController@updateItem')->name('customer.cart.update');
        Route::delete('/item/{id}', 'CartController@removeItem')->name('customer.cart.remove');
        Route::delete('/clear', 'CartController@clear')->name('customer.cart.clear');
        Route::post('/coupon', 'CartController@applyCoupon')->name('customer.cart.apply-coupon');
        Route::delete('/coupon', 'CartController@removeCoupon')->name('customer.cart.remove-coupon');
    });

    // Orders
    Route::prefix('orders')->group(function () {
        Route::get('/', 'OrderController@index')->name('customer.orders.index');
        Route::post('/', 'OrderController@store')->name('customer.orders.store');
        Route::get('/active', 'OrderController@active')->name('customer.orders.active');
        Route::get('/{uuid}', 'OrderController@show')->name('customer.orders.show');
        Route::get('/{uuid}/track', 'OrderController@track')->name('customer.orders.track');
        Route::post('/{uuid}/cancel', 'OrderController@cancel')->name('customer.orders.cancel');
        Route::post('/{uuid}/review', 'OrderController@review')->name('customer.orders.review');
        Route::post('/{uuid}/reorder', 'OrderController@reorder')->name('customer.orders.reorder');
    });

    // Addresses
    Route::prefix('addresses')->group(function () {
        Route::get('/', 'AddressController@index')->name('customer.addresses.index');
        Route::post('/', 'AddressController@store')->name('customer.addresses.store');
        Route::get('/{id}', 'AddressController@show')->name('customer.addresses.show');
        Route::put('/{id}', 'AddressController@update')->name('customer.addresses.update');
        Route::delete('/{id}', 'AddressController@destroy')->name('customer.addresses.destroy');
        Route::post('/{id}/default', 'AddressController@setDefault')->name('customer.addresses.set-default');
    });

    // Wallet
    Route::prefix('wallet')->group(function () {
        Route::get('/', 'WalletController@show')->name('customer.wallet.show');
        Route::get('/transactions', 'WalletController@transactions')->name('customer.wallet.transactions');
        Route::post('/topup', 'WalletController@topup')->name('customer.wallet.topup');
    });

    // Favorites
    Route::prefix('favorites')->group(function () {
        Route::get('/restaurants', 'FavoriteController@restaurants')->name('customer.favorites.restaurants');
        Route::get('/items', 'FavoriteController@items')->name('customer.favorites.items');
    });

    // Chat
    Route::prefix('chat')->group(function () {
        Route::get('/conversations', 'ChatController@conversations')->name('customer.chat.conversations');
        Route::get('/conversations/{uuid}', 'ChatController@show')->name('customer.chat.show');
        Route::post('/conversations/{uuid}/messages', 'ChatController@sendMessage')->name('customer.chat.send');
    });

    // Notifications
    Route::prefix('notifications')->group(function () {
        Route::get('/', 'NotificationController@index')->name('customer.notifications.index');
        Route::post('/{id}/read', 'NotificationController@markAsRead')->name('customer.notifications.read');
        Route::post('/read-all', 'NotificationController@markAllAsRead')->name('customer.notifications.read-all');
    });
});

// Public routes (no auth required)
Route::prefix('public')->group(function () {
    Route::get('/zones', 'ZoneController@index')->name('customer.zones.index');
    Route::get('/zones/{slug}/campuses', 'ZoneController@campuses')->name('customer.zones.campuses');
    Route::get('/campuses/{slug}/locations', 'CampusController@locations')->name('customer.campuses.locations');
    Route::get('/restaurant-categories', 'RestaurantController@allCategories')->name('customer.restaurant-categories');
    Route::get('/delivery-fee', 'DeliveryController@calculate')->name('customer.delivery-fee');
});
