<?php

use Illuminate\Support\Facades\Route;

/*
|--------------------------------------------------------------------------
| Vendor API Routes
|--------------------------------------------------------------------------
|
| Routes for vendor/restaurant management.
| All routes are prefixed with /api/v1/vendor
|
*/

// Vendor Registration (guest)
Route::post('/register', 'VendorAuthController@register')
    ->middleware(['throttle:auth'])
    ->name('vendor.register');

// Authenticated vendor routes
Route::middleware(['auth:sanctum', 'user.type:vendor'])->group(function () {
    // Dashboard
    Route::get('/dashboard', 'DashboardController@index')->name('vendor.dashboard');
    Route::get('/dashboard/stats', 'DashboardController@stats')->name('vendor.dashboard.stats');

    // Profile & KYC
    Route::prefix('profile')->group(function () {
        Route::get('/', 'ProfileController@show')->name('vendor.profile.show');
        Route::put('/', 'ProfileController@update')->name('vendor.profile.update');
        Route::post('/documents', 'ProfileController@uploadDocument')->name('vendor.profile.upload-document');
        Route::get('/documents', 'ProfileController@documents')->name('vendor.profile.documents');
    });

    // Restaurant Management
    Route::prefix('restaurant')->group(function () {
        Route::get('/', 'RestaurantController@show')->name('vendor.restaurant.show');
        Route::put('/', 'RestaurantController@update')->name('vendor.restaurant.update');
        Route::post('/logo', 'RestaurantController@uploadLogo')->name('vendor.restaurant.upload-logo');
        Route::post('/cover', 'RestaurantController@uploadCover')->name('vendor.restaurant.upload-cover');
        Route::put('/hours', 'RestaurantController@updateHours')->name('vendor.restaurant.update-hours');
        Route::post('/toggle-status', 'RestaurantController@toggleStatus')->name('vendor.restaurant.toggle-status');
    });

    // Menu Management
    Route::prefix('menu')->group(function () {
        // Categories
        Route::get('/categories', 'MenuCategoryController@index')->name('vendor.menu.categories.index');
        Route::post('/categories', 'MenuCategoryController@store')->name('vendor.menu.categories.store');
        Route::put('/categories/{id}', 'MenuCategoryController@update')->name('vendor.menu.categories.update');
        Route::delete('/categories/{id}', 'MenuCategoryController@destroy')->name('vendor.menu.categories.destroy');
        Route::post('/categories/reorder', 'MenuCategoryController@reorder')->name('vendor.menu.categories.reorder');

        // Items
        Route::get('/items', 'MenuItemController@index')->name('vendor.menu.items.index');
        Route::post('/items', 'MenuItemController@store')->name('vendor.menu.items.store');
        Route::get('/items/{uuid}', 'MenuItemController@show')->name('vendor.menu.items.show');
        Route::put('/items/{uuid}', 'MenuItemController@update')->name('vendor.menu.items.update');
        Route::delete('/items/{uuid}', 'MenuItemController@destroy')->name('vendor.menu.items.destroy');
        Route::post('/items/{uuid}/image', 'MenuItemController@uploadImage')->name('vendor.menu.items.upload-image');
        Route::post('/items/{uuid}/toggle-availability', 'MenuItemController@toggleAvailability')->name('vendor.menu.items.toggle');
        Route::put('/items/{uuid}/stock', 'MenuItemController@updateStock')->name('vendor.menu.items.stock');

        // Variants
        Route::post('/items/{uuid}/variant-groups', 'VariantController@storeGroup')->name('vendor.menu.variants.store-group');
        Route::put('/variant-groups/{id}', 'VariantController@updateGroup')->name('vendor.menu.variants.update-group');
        Route::delete('/variant-groups/{id}', 'VariantController@destroyGroup')->name('vendor.menu.variants.destroy-group');
        Route::post('/variant-groups/{id}/variants', 'VariantController@storeVariant')->name('vendor.menu.variants.store');
        Route::put('/variants/{id}', 'VariantController@updateVariant')->name('vendor.menu.variants.update');
        Route::delete('/variants/{id}', 'VariantController@destroyVariant')->name('vendor.menu.variants.destroy');

        // Addons
        Route::post('/items/{uuid}/addons', 'AddonController@store')->name('vendor.menu.addons.store');
        Route::put('/addons/{id}', 'AddonController@update')->name('vendor.menu.addons.update');
        Route::delete('/addons/{id}', 'AddonController@destroy')->name('vendor.menu.addons.destroy');
    });

    // Orders
    Route::prefix('orders')->group(function () {
        Route::get('/', 'OrderController@index')->name('vendor.orders.index');
        Route::get('/pending', 'OrderController@pending')->name('vendor.orders.pending');
        Route::get('/active', 'OrderController@active')->name('vendor.orders.active');
        Route::get('/{uuid}', 'OrderController@show')->name('vendor.orders.show');
        Route::post('/{uuid}/accept', 'OrderController@accept')->name('vendor.orders.accept');
        Route::post('/{uuid}/reject', 'OrderController@reject')->name('vendor.orders.reject');
        Route::post('/{uuid}/preparing', 'OrderController@markPreparing')->name('vendor.orders.preparing');
        Route::post('/{uuid}/ready', 'OrderController@markReady')->name('vendor.orders.ready');
    });

    // Finance
    Route::prefix('finance')->group(function () {
        Route::get('/summary', 'FinanceController@summary')->name('vendor.finance.summary');
        Route::get('/settlements', 'FinanceController@settlements')->name('vendor.finance.settlements');
        Route::get('/settlements/{uuid}', 'FinanceController@settlementDetails')->name('vendor.finance.settlement-details');
        Route::get('/transactions', 'FinanceController@transactions')->name('vendor.finance.transactions');
    });

    // Analytics
    Route::prefix('analytics')->group(function () {
        Route::get('/sales', 'AnalyticsController@sales')->name('vendor.analytics.sales');
        Route::get('/orders', 'AnalyticsController@orders')->name('vendor.analytics.orders');
        Route::get('/items', 'AnalyticsController@topItems')->name('vendor.analytics.items');
        Route::get('/customers', 'AnalyticsController@customers')->name('vendor.analytics.customers');
    });

    // Reviews
    Route::prefix('reviews')->group(function () {
        Route::get('/', 'ReviewController@index')->name('vendor.reviews.index');
        Route::post('/{id}/reply', 'ReviewController@reply')->name('vendor.reviews.reply');
    });

    // Chat
    Route::prefix('chat')->group(function () {
        Route::get('/conversations', 'ChatController@conversations')->name('vendor.chat.conversations');
        Route::get('/conversations/{uuid}', 'ChatController@show')->name('vendor.chat.show');
        Route::post('/conversations/{uuid}/messages', 'ChatController@sendMessage')->name('vendor.chat.send');
    });

    // Notifications
    Route::prefix('notifications')->group(function () {
        Route::get('/', 'NotificationController@index')->name('vendor.notifications.index');
        Route::post('/{id}/read', 'NotificationController@markAsRead')->name('vendor.notifications.read');
        Route::post('/read-all', 'NotificationController@markAllAsRead')->name('vendor.notifications.read-all');
    });
});
